/*
Collision Mask Utilities
Copyright ©2014 David Powell

Released under the MIT Licence

Permission is hereby granted, free of charge, to any person obtaining a copy
of this software and associated documentation files (the "Software"), to deal
in the Software without restriction, including without limitation the rights
to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
copies of the Software, and to permit persons to whom the Software is
furnished to do so, subject to the following conditions:

The above copyright notice and this permission notice shall be included in
all copies or substantial portions of the Software.

THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN
THE SOFTWARE.
*/

#include "mainwindow.h"
#include "ui_mainwindow.h"

const QString ApplicationName("Collision Mask Viewer");
const QString ApplicationVersion("V1.0");

/*!
 * \brief MainWindow constructor
 * \param parent The parent widget
 *
 * Initialise the application, optionally loading a mask file
 * if specified.
 */
MainWindow::MainWindow(QString filename, QWidget *parent) :
    QMainWindow(parent),
    ui(new Ui::MainWindow),
    zoom(1)
{
    ui->setupUi(this);

    qApp->installEventFilter(this);

    imageLabel.setSizePolicy(QSizePolicy::Ignored, QSizePolicy::Ignored);
    imageLabel.setScaledContents(true);
    QPalette palette = imageLabel.palette();
    QColor color = ui->scrollArea->palette().background().color();
    palette.setColor(imageLabel.backgroundRole(), color);
    imageLabel.setPalette(palette);

    ui->scrollArea->setWidget(&imageLabel);
    ui->statusBar->showMessage(tr("Width: %1 Height: %2").arg("0").arg("0"));

    if (filename.isEmpty() == false)
    {
        loadMask(filename);
    }
}

/*!
 * \brief MainWindow destructor
 */
MainWindow::~MainWindow()
{
    delete ui;
}

/*!
 * \brief Suppress changing the status tip automatically
 * \param event The event to check
 * \return true if the event should be filtered
 */
bool MainWindow::eventFilter(QObject * /* watched */, QEvent *event)
{
    return event->type() == QEvent::StatusTip;
}

/*!
 * \brief When the user drags an item; only allow files
 * \param event The drag event to check
 */
void MainWindow::dragEnterEvent(QDragEnterEvent *event)
{
    if (event->mimeData()->hasFormat("text/uri-list") == true)
    {
        event->acceptProposedAction();
    }
}

/*!
 * \brief When a file is dropped on the application;
 *        check it is a CMSK file and then load it.
 * \param event The drop event to use
 */
void MainWindow::dropEvent(QDropEvent *event)
{
    QList<QUrl> urls = event->mimeData()->urls();
    if (urls.isEmpty() == true)
    {
        return;
    }

    QString filename = urls.first().toLocalFile();
    if (filename.isEmpty() == true)
    {
        return;
    }

    QFileInfo fileInfo(filename);
    if (fileInfo.suffix().toLower() != "cmsk")
    {
        return;
    }

    loadMask(filename);
}

/*!
 * \brief When the exit action is triggered; close the window
 *        thereby ending the application.
 */
void MainWindow::on_actionExit_triggered()
{
    this->close();
}

/*!
 * \brief When the about action is triggered;
 *        display the about box.
 */
void MainWindow::on_actionAbout_triggered()
{
    QMessageBox::information(this, ApplicationName,
                             ApplicationName + " " + ApplicationVersion + "\n" +
                             tr("Copyright ©2014 David Powell.\n") +
                             tr("Released under the MIT License.\n\n") +
                             tr("http://www.loadcode.co.uk"),
                             QMessageBox::Ok);
}

/*!
 * \brief When the check for updates action is triggered;
 *        Open the http://www.loadcode.co.uk URL
 */
void MainWindow::on_actionCheck_for_Updates_triggered()
{
    QDesktopServices::openUrl(QUrl("http://www.loadcode.co.uk"));
}

/*!
 * \brief When the open action is triggered;
 *        Ask the user for a mask file and then open it.
 */
void MainWindow::on_actionOpen_triggered()
{
    QFileDialog dialog(this);
    dialog.setFileMode(QFileDialog::ExistingFile);

    QStringList filters;
    filters << tr("Collision Mask Files") + " (*.cmsk)"
#ifdef Q_OS_WIN32
            << tr("All Files") + " (*.*)";
#else
            << tr("All Files") + " (*)";
#endif
   dialog.setNameFilters(filters);

    QStringList filenames;
    if (dialog.exec() != QDialog::Accepted)
    {
        return;
    }

    filenames = dialog.selectedFiles();

    loadMask(filenames[0]);
}

/*!
 * \brief Load the specified mask file
 * \param filename The mask file to load
 */
void MainWindow::loadMask(QString filename)
{
    mask = std::unique_ptr<Mask>(new Mask());

    try
    {
        mask->load(filename);
    }
    catch (std::exception const& ex)
    {
        mask.reset();
        QMessageBox::information(this, ApplicationName,
                                 ex.what(),
                                 QMessageBox::Ok);
    }

    updateImage();
}

/*!
 * \brief Update the pixmap from the data in the
 *        currently loaded mask and then use that to
 *        update the imageLabel
 */
void MainWindow::updateImage()
{
    if (!mask)
    {
        QImage image(0, 0, QImage::Format_RGB32);
        imageLabel.setPixmap(QPixmap::fromImage(image));
        imageLabel.resize(imageLabel.pixmap()->size());
        ui->statusBar->showMessage(tr("Width: %1 Height: %2").arg("0").arg("0"));
        return;
    }

    ui->statusBar->showMessage(tr("Width: %1 Height: %2").arg(mask->width).arg(mask->height));

    QImage image(mask->width, mask->height, QImage::Format_RGB32);

    quint32 heightOffset = 0;

    for (quint32 y = 0; y < mask->height; y++)
    {
        for (quint32 x = 0; x < mask->width; x++)
        {
            quint8 colour = mask->data[heightOffset + x];
            image.setPixel(x, y, qRgb(colour, colour, colour));
        }

        heightOffset += mask->width;
    }

    zoom = 1;

    pixmap = QPixmap::fromImage(image);

    scaledPixmap = pixmap.copy(pixmap.rect());
    imageLabel.setPixmap(scaledPixmap);
    imageLabel.resize(imageLabel.pixmap()->size());
}

/*!
 * \brief When the zoom in action is triggered;
 *        increase the zoom, scale the pixmap by the zoom amount
 *        and update the imageLabel
 */
void MainWindow::on_actionZoom_In_triggered()
{
    if (pixmap.isNull() == false)
    {
        zoom++;
        scaledPixmap = pixmap.scaled(pixmap.width() * zoom, pixmap.height() * zoom, Qt::IgnoreAspectRatio, Qt::FastTransformation);
        imageLabel.setPixmap(scaledPixmap);
        imageLabel.resize(imageLabel.pixmap()->size());
    }
}

/*!
 * \brief When the zoom out action is triggered;
 *        decrease the zoom, scale the pixmap by the zoom amount
 *        and update the imageLabel
 */
void MainWindow::on_actionZoom_Out_triggered()
{
    if (pixmap.isNull() == false)
    {
        if (zoom > 1)
        {
            zoom--;
            scaledPixmap = pixmap.scaled(pixmap.width() * zoom, pixmap.height() * zoom, Qt::IgnoreAspectRatio, Qt::FastTransformation);
            imageLabel.setPixmap(scaledPixmap);
            imageLabel.resize(imageLabel.pixmap()->size());
        }
    }
}
